<?php

namespace App\Http\Controllers;

use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Http\Response;

class ServiceController extends Controller
{


    public function index()
    {
        try {
            $services = Service::with(['serviceRoles', 'servicePermissions.users'])->get();

            $formattedServices = $services->map(function ($service) {
                return [
                    'id' => $service->id,
                    'name' => $service->name,
                    'service_id' => $service->service_id,
                    'description' => $service->descriptions,
                    'color' => $service->color,
                    'root_url' => $service->root_url,
                    'service_roles' => $service->serviceRoles->map(fn($role) => [
                        'id' => $role->id,
                        'role' => $role->role
                    ]),
                    'service_permissions' => $service->servicePermissions->map(fn($perm) => [
                        'id' => $perm->id,
                        'permission' => $perm->permission
                    ]),
                    'users_count' => $service->servicePermissions->flatMap->users->count(),
                    'permissions_count' => $service->servicePermissions->count() ?: 0,
                    'roles_count'       => $service->serviceRoles->count() ?: 0
                ];
            });

            return response()->json([
                'status' => 'success',
                'message' => 'Services retrieved successfully',
                'data' => $formattedServices
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve services',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'service_id' => 'required|integer|unique:services',
            'descriptions' => 'required|string',

        ], [
            'name.required' => 'Name is requied',
            'service_id.unique' => 'service_id is already registered',
            'service_id.required' => 'service_ id is required',
            'descriptions.required' => ' description is required'
        ]);

        $service_uuid = Str::uuid();

        $service = Service::create([
            'name' => $validated['name'],
            'descriptions' => $validated['descriptions'],
            'service_id' => $validated['service_id'],
            'uuid' => $service_uuid
        ]);
        return response()->json($service, 201);
    }

    public function show($id)
    {
        $service = Service::with(['serviceRoles', 'servicePermissions',])->findOrFail($id);
        return response()->json($service);
    }

    public function update(Request $request, $id)
    {
        $service = Service::findOrFail($id);
        $service->update($request->all());
        return response()->json($service);
    }

    public function destroy($id)
    {
        Service::destroy($id);
        return response()->json([
            'success' => true,
            'message' => 'Service deleted successfully'
        ]);
    }
}
