<?php

namespace App\Http\Controllers\Auth;

use Illuminate\Http\Request;
use App\Models\User;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use Illuminate\Support\Facades\Hash;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use App\Models\UserSession;
use App\Models\ServicePermission;
use Illuminate\Support\Facades\Log;
use Jenssegers\Agent\Agent;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    // Token expiration time in seconds
    const TOKEN_EXPIRATION = 3600; // 1 hour

    public function register(Request $request)
    {
        $validated = $request->validate(
            [
                'username' => 'required|string|max:255',
                'email' => 'required|email|unique:users',
                'phone_number' => 'required|numeric|unique:users',
                'password' => 'required|min:8',
            ],
            [
                // Custom error messages
                'username.required' => 'Username is required',
                'email.unique' => 'This email is already registered',
                'phone_number.unique' => 'This phone number is already in use',
                'password.min' => 'Password must be at least 8 characters',
            ]
        );
        $user = User::create([
            'username' => $validated['username'],
            'email' => $validated['email'],
            'phone_number' => $validated['phone_number'],
            'password' => Hash::make($validated['password'])
        ]);
        return response()->json([
            'message' => 'User registered successfully',
            'user' => $user->only(['id', 'username', 'email', 'phone_number'])
        ], 201);
    }

    public function login(Request $request)
    {
        $validated = $request->validate([
            'login' => 'required',
            'password' => 'required',
            'device_id' => 'required|string',
            'device_type' => 'sometimes|in:web,mobile'
        ]);

        try {
            $user = User::where(function ($query) use ($validated) {
                $query->whereRaw('BINARY username = ?', [$validated['login']])
                    ->orWhereRaw('BINARY phone_number = ?', [$validated['login']]);
            })
                ->with([
                    'servicePermissions.service',
                    'companyUserId' // Now singular (one-to-one)
                ])
                ->firstOrFail();

            if (!Hash::check($validated['password'], $user->password)) {
                throw new \Exception('Invalid credentials', 401);
            }

            // Device detection
            $agent = new Agent();
            $deviceType = $validated['device_type'] ??
                ($agent->isMobile() ? 'mobile' : 'web');

            // Session limit check            
            $sessionCount = UserSession::where('user_id', $user->id)
                ->where('device_type', $deviceType)
                ->where('is_active', 1)
                ->count();

            // if ($sessionCount >= 2) {
            //     return response()->json([
            //         'success' => "error",
            //         'message' => "Maximum 2 active {$deviceType} sessions reached",
            //         'max_sessions' => 2,
            //         'current_sessions' => $sessionCount
            //     ], 403);
            // }
            // Create or update session

            $tokenId = Str::uuid();

            $session = UserSession::create([
                'user_id' => $user->id,
                'device_id' => $validated['device_id'],
                'device_type' => $deviceType,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'last_activity' => now(),
                'is_active' => 1,
                'token_id' => $tokenId
            ]);

            // Generate JWT token       

          

            $token = $this->generateJwtToken($user, $session);
              
            
            
            return response()->json([
                'success' => true,
                'access_token' => @$token,
                'token_type' => 'Bearer',
                'expires_in' => self::TOKEN_EXPIRATION,
                'user' => $user->only(['id', 'username', 'email', 'phone_number']),
                'services' => $this->prepareServicesData($user),
                'session_info' => [
                    'device_id' => $validated['device_id'],
                    'device_type' => $deviceType,
                    'current_sessions' => (int) ($sessionCount + 1), // Cast to int
                    'max_sessions' => 2
                ],
                'permissions' => $user->servicePermissions->pluck('permission')->unique()->values()
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json(['message' => 'User not found'], 404);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'code' => $e->getCode()
            ], 500);
        }
    }

    public function logout(Request $request)
    {
        try {
            $token = $request->bearerToken();
            $secret = (string) env('JWT_SECRET');
            
            $payload = JWT::decode($token, new Key($secret ,'HS256'));

            // Mark session as inactive
            UserSession::where('user_id', $payload->sub)
                ->where('token_id', $payload->session_id)
                ->update(['is_active' => 0]);

            return response()->json(['message' => 'Successfully logged out']);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Logout failed'], 500);
        }
    }

    public function refreshToken(Request $request)
    {
        try {
            $oldToken = $request->bearerToken();
            $payload = JWT::decode($oldToken, new Key(env('JWT_SECRET'), 'HS256'));

            // Verify token is expired but not too old
            if ($payload->exp > time() - 3600) { // Within 1 hour of expiration
                throw new \Exception('Token not expired', 400);
            }

            $user = User::findOrFail($payload->sub);
            $session = UserSession::where('token_id', $payload->session_id)
                ->firstOrFail();

            // Generate new token with same session
            $newToken = $this->generateJwtToken($user, $session);

            return response()->json([
                'access_token' => $newToken,
                'token_type' => 'Bearer',
                'expires_in' => self::TOKEN_EXPIRATION
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Token refresh failed: ' . $e->getMessage()
            ], 401);
        }
    }

    public function getUserServicePermissions(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|integer|exists:users,id',
            'service_id' => 'required|integer|exists:services,id'
        ]);

        $permissions = ServicePermission::where('service_id', $validated['service_id'])
            ->whereHas('users', function ($query) use ($validated) {
                $query->where('users.id', $validated['user_id']);
            })
            ->get(['permission']);

        return response()->json([
            'success' => true,
            'message' => "permission fetched Successfully",
            'data' => [
                'user_id' => $validated['user_id'],
                'service_id' => $validated['service_id'],
                'permissions' => $permissions
            ]
        ], 200);
    }


    public function allSessionStatuses()
    {

        // Load all sessions with related user
        $sessions = UserSession::with('user')->get()->map(function ($session) {
            return [
                'session_id' => $session->id,
                'user_id' => $session->user_id,
                'username' => $session->user->username ?? null,
                'email' => $session->user->email ?? null,
                'phone_number' => $session->user->phone_number ?? null,
                'device_id' => $session->device_id,
                'device_type' => $session->device_type,
                'is_active' => $session->is_active,
                'last_login' => $session->created_at,
                'last_activity' => $session->last_activity,
                'last_login' => $session->created_at,
                'status' => $session->is_active == 1 ? 'Active' : 'Inactive',
                'ip' => $session->ip_address,
                'user_agent' => $session->user_agent,
            ];
        });
        return response()->json([
            'success' => true,
            'sessions' => $sessions
        ]);
    }




    /**
     * Helper Methods
     */

    protected function generateJwtToken(User $user, UserSession $session): string
    {
        $payload = [
            'sub' => $user->id,
            'name' => $user->username,
            'email' => $user->email,
            'session_id' => $session->token_id,
            'device_id' => $session->device_id,
            'iat' => time(),
            'jti' => Str::uuid() // Unique token identifier
        ];

        $secret = (string) env('JWT_SECRET');
       return JWT::encode($payload, $secret, 'HS256');
        // return  env('JWT_SECRET');
    }


    protected function prepareServicesData(User $user): array
    {
        return $user->servicePermissions
            ->groupBy('service_id')
            ->map(function ($permissions, $serviceId) {
                $service = $permissions->first()->service;
                return [
                    'service_id' => $service->id,
                    'name' => $service->name,
                    //  'permissions' => $permissions->pluck('permission')->toArray()
                ];
            })
            ->values()
            ->toArray();
    }

    public function validateSession(Request $request)
    {
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'message' => 'Missing token',
                'code' => 401,
            ], 401);
        }

        try {
            // Decode JWT token
            $decoded = JWT::decode($token, new Key(env('JWT_SECRET'), 'HS256'));

            // Return basic user info from token
            return response()->json([
                'user' => [
                    'id' => $decoded->sub ?? null,
                    'email' => $decoded->email ?? null,
                    'name' => $decoded->name ?? null,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Invalid or expired token',
                'code' => 401,
                'error' => $e->getMessage(),
            ], 401);
        }
    }

    public function updatePassword(Request $request)
    {
        // Validate input
        $validator = Validator::make($request->all(), [
            'old_password' => 'required|string',
            'new_password' => 'required|string|min:6|confirmed',
            // field must be: new_password_confirmation
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors(),
            ], 422);
        }

        try {
            // Get logged-in user from JWT
            $token = $request->bearerToken();
            $secret = (string) env('JWT_SECRET');
            $payload = JWT::decode($token, new Key($secret, 'HS256'));
            $user = User::findOrFail($payload->sub);

            // Check old password
            if (!Hash::check($request->old_password, $user->password)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Old password is incorrect'
                ], 400);
            }

            // Update new password
            $user->password = Hash::make($request->new_password);

            $user->save();

            // OPTIONAL: logout all sessions except this one
            UserSession::where('user_id', $user->id)
                ->where('token_id', '!=', $payload->session_id)
                ->update(['is_active' => 0]);

            return response()->json([
                'success' => true,
                'message' => 'Password updated successfully'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }
}
